using System;
using System.Collections;
using System.Data;
using System.Reflection;
using Method = System.Reflection.MethodBase;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>Equipment business layer class</summary>

		#endregion
	
	
	public class Equipment:BaseBusinessObject
	{

		#region Variables

		private	const string MIN_COLUMNS = TABLE.Equipment.Comments + "," + TABLE.Equipment.DivisionCode + "," + TABLE.Equipment.EquipmentGuid + "," + TABLE.Equipment.EquipmentName + "," + 
											TABLE.Equipment.InspectionIndicator + "," + TABLE.Equipment.EquipmentType + "," + TABLE.Equipment.InvoiceNumber + "," + TABLE.Equipment.LocalIdentifier + "," + TABLE.Equipment.MaintenanceContractDate + "," + 
											TABLE.Equipment.MaintenanceContractNumber + "," + TABLE.Equipment.ManufacturerName + "," + TABLE.Equipment.ReceiptDate + "," + 
											TABLE.Equipment.ReceiptTechId + "," + TABLE.Equipment.RowVersion + "," + TABLE.Equipment.SerialNumber;
			
			
		private System.Data.DataRow _drEquipmentRow;
		private System.Data.DataTable _dtEquipment;
		private BOL.Address _manufacturerAddress = null;
		private bool _isMaintenanceContractDate = false;

		#endregion


		#region Constructors

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="59"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>Valid Equipment Object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="60"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Equipment object constructor - populates with data from guid
		/// </summary>
		public Equipment(System.Guid equipmentGUID)
		{
			if (equipmentGUID == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("equipment guid").ResString);
			}
			//
			//Get the datarow for the requested guid
			this._dtEquipment = DAL.Equipment.GetEquipmentByGuid(equipmentGUID);
			//this._equipmentMaintenance = new BOL.EquipmentMaintenance(equipmentGUID);
			if (this._dtEquipment.Rows.Count == 0)
			{
				this._drEquipmentRow = this._dtEquipment.NewRow();
				this._drEquipmentRow[TABLE.Equipment.EquipmentGuid] = equipmentGUID;
				this._drEquipmentRow[TABLE.Address.AddressGuid] = Guid.NewGuid();
				this.IsNew = true;
				this.IsDirty = true;
				this.SetDefaultsInTable();
				this._manufacturerAddress = new BOL.Address(this._drEquipmentRow);
				this._manufacturerAddress.IsNew = true;
			}
			else
			{
				this._drEquipmentRow = this._dtEquipment.Rows[0];
				//Address needs to be loaded first because of LoadFromDataRow
				this._manufacturerAddress = new BOL.Address(_drEquipmentRow);
				this.LoadFromDataRow(_drEquipmentRow);
				this.IsDirty = false;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="61"> 
		///		<ExpectedInput>Valid equipment data row</ExpectedInput>
		///		<ExpectedOutput>Valid Equipement Object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="62"> 
		///		<ExpectedInput>Invalid Data row</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Equipment object constructor - populates with data from data row
		/// </summary>
		/// <param name="dtRow">Equipment data row</param>
		public Equipment(System.Data.DataRow dtRow)
		{
			if (!Common.Utility.VerifyMinColumnsInDataTable(MIN_COLUMNS.Split(','), dtRow.Table))
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(TABLE.Equipment.TableName).ResString);

			//Because the address stuff is FOOBAR, this line needs to be done first
			this._manufacturerAddress = new BOL.Address(dtRow);
			this._dtEquipment = DAL.Equipment.GetEmptyEquipmentTable();
			
			this.LoadFromDataRow(dtRow);

			//this._equipmentMaintenance = new BOL.EquipmentMaintenance(this.EquipmentGuid);

		}
		
		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="63"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>EquipmentName property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="64"> 
		///		<ExpectedInput>Invalid string (76 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Equipment Name
		/// </summary>
		public string EquipmentName
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.EquipmentName))
		
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.EquipmentName].ToString();
			}
			set
			{
				if (value.Length > 75)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "75").ResString;
					throw new ArgumentException(errorMessage);
				}
				//
				this._drEquipmentRow[TABLE.Equipment.EquipmentName] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 75)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="65"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>ReceiptDate property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="66"> 
		///		<ExpectedInput>Invalid Date/time (future date/time)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Receipt Date/Time
		/// </summary>
		public System.DateTime ReceiptDate
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.ReceiptDate))
					return System.DateTime.MinValue;

				return (System.DateTime)this._drEquipmentRow[TABLE.Equipment.ReceiptDate];
			}
			set
			{
				if (value > BOL.VBECSDateTime.GetDivisionCurrentDateTime())
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);

				this._drEquipmentRow[TABLE.Equipment.ReceiptDate] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value == System.DateTime.MinValue)||(value>BOL.VBECSDateTime.GetDivisionCurrentDateTime())));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="67"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>ReceiptTechId property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="68"> 
		///		<ExpectedInput>Invalid string (31 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Receipt Tech Id
		/// </summary>
		public string ReceiptTechId
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.ReceiptTechId))
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.ReceiptTechId].ToString();
			}
			set
			{
				if (value.Length > 30)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "30").ResString;
					throw new ArgumentException(errorMessage);
				}
				
				this._drEquipmentRow[TABLE.Equipment.ReceiptTechId] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 30)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="69"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>InvoiceNumber property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="70"> 
		///		<ExpectedInput>Invalid String (21 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Invoice Number
		/// </summary>
		public string InvoiceNumber
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.InvoiceNumber))
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.InvoiceNumber].ToString();
			}
			set
			{
				if (value.Length > 20)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "20").ResString;
					throw new ArgumentException(errorMessage);
				}
		
				this._drEquipmentRow[TABLE.Equipment.InvoiceNumber] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 20)));
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="71"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>LocalIdentifier property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="72"> 
		///		<ExpectedInput>Invalid String (26 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Local Identifier
		/// </summary>
		public string LocalIdentifier
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.LocalIdentifier))
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.LocalIdentifier].ToString();
			}
			set
			{
				if (value.Length > 25)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "25").ResString;
					throw new ArgumentException(errorMessage);
				}
				
				this._drEquipmentRow[TABLE.Equipment.LocalIdentifier] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 25)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="73"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>MaintenanceContractDate property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="74"> 
		///		<ExpectedInput>Invalid date (Min Date)</ExpectedInput>
		///		<ExpectedOutput>Rule broken</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Maintenance Contract Expiration/Renewal Date
		/// </summary>
		public System.DateTime MaintenanceContractDate
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.MaintenanceContractDate))
					return System.DateTime.MinValue;

				return (System.DateTime)this._drEquipmentRow[TABLE.Equipment.MaintenanceContractDate];
			}
			set
			{
				this._drEquipmentRow[TABLE.Equipment.MaintenanceContractDate] = value;
				this.IsDirty = true;
				//
				RuleBroken(Common.VbecsBrokenRules.FrmMaintainEquipment.MaintenanceContractDate, ((this._isMaintenanceContractDate) && (value==DateTime.MinValue)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5630"> 
		///		<ExpectedInput>Boolean</ExpectedInput>
		///		<ExpectedOutput>IsMaintenanceContractDate property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5631"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets indicator of whether a maintenance contract renewal/expiration date exists
		/// </summary>
		public bool IsMaintenanceContractDate
		{
			get
			{
				return this._isMaintenanceContractDate;
			}
			set
			{
				this._isMaintenanceContractDate = value;
				this.IsDirty = true;
				//
				RuleBroken(Common.VbecsBrokenRules.FrmMaintainEquipment.MaintenanceContractDate, ((value) && (this.MaintenanceContractDate==DateTime.MinValue)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="75"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>MaintenanceContractNumber property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="76"> 
		///		<ExpectedInput>Invalid string (26 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Maintenance Contract Number
		/// </summary>
		public string MaintenanceContractNumber
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.MaintenanceContractNumber))
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.MaintenanceContractNumber].ToString();
			}
			set
			{
				if (value.Length > 25)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "25").ResString;
					throw new ArgumentException(errorMessage);
				}

				this._drEquipmentRow[TABLE.Equipment.MaintenanceContractNumber] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 25)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="77"> 
		///		<ExpectedInput>Boolean</ExpectedInput>
		///		<ExpectedOutput>InspectionIndicator property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="78"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Inspection Indicator (true = satisfactory, false = unsatisfactory)
		/// </summary>
		public bool InspectionIndicator
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.InspectionIndicator))
					return false;

				return (bool)this._drEquipmentRow[TABLE.Equipment.InspectionIndicator];
			}
			set
			{
				this._drEquipmentRow[TABLE.Equipment.InspectionIndicator] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5632"> 
		///		<ExpectedInput>Boolean</ExpectedInput>
		///		<ExpectedOutput>OutOfServiceIndicator property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5633"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Out-Of-Service Indicator
		/// </summary>
		public bool OutOfServiceIndicator
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.OutOfServiceIndicator))
					return false;

				return (bool)this._drEquipmentRow[TABLE.Equipment.OutOfServiceIndicator];
			}
			set
			{
				this._drEquipmentRow[TABLE.Equipment.OutOfServiceIndicator] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="79"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Comments property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="80"> 
		///		<ExpectedInput>Invalid string (256 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Comments
		/// </summary>
		public override string Comments
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.Comments))
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.Comments].ToString();
			}
			set
			{
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "255").ResString;
					throw new ArgumentException(errorMessage);
				}

				this._drEquipmentRow[TABLE.Equipment.Comments] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 255)));
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="81"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>EquipmentGuid property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="82"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets Equipment Guid (Read-Only, should be set during MakeNew() of class)
		/// </summary>
		public Guid EquipmentGuid
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.EquipmentGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drEquipmentRow[TABLE.Equipment.EquipmentGuid];
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="83"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>SerialNumber property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="84"> 
		///		<ExpectedInput>Invalid string (51 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Serial Number
		/// </summary>
		public string SerialNumber
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.SerialNumber))
				{
					return string.Empty;
				}
                return this._drEquipmentRow[TABLE.Equipment.SerialNumber].ToString();
			}
			set
			{
				if (value.Length > 50)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											MethodBase.GetCurrentMethod().Name, true), "50").ResString;
					throw new ArgumentException(errorMessage);
				}
				this._drEquipmentRow[TABLE.Equipment.SerialNumber] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 50)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="85"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>EquipmentType property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="86"> 
		///		<ExpectedInput>Valid string (26 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Equipment Type
		/// </summary>
		public string EquipmentType
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.EquipmentType))
					return string.Empty;
				
				return this._drEquipmentRow[TABLE.Equipment.EquipmentType].ToString();
			}
			set
			{
				if (value.Length > 25)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											MethodBase.GetCurrentMethod().Name, true), "25").ResString;
					throw new ArgumentException(errorMessage);
				}
				this._drEquipmentRow[TABLE.Equipment.EquipmentType] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 25)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="87"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>ManufacturerName propert</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="88"> 
		///		<ExpectedInput>Invalid string (51 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Manufacturer Name
		/// </summary>
		public string ManufacturerName
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.ManufacturerName))
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.ManufacturerName].ToString();
			}
			set
			{
				if (value.Length > 50)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											MethodBase.GetCurrentMethod().Name, true), "50").ResString;
					throw new ArgumentException(errorMessage);
				}
				this._drEquipmentRow[TABLE.Equipment.ManufacturerName] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 50)));
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5634"> 
		///		<ExpectedInput>Valid Address object</ExpectedInput>
		///		<ExpectedOutput>ManufacturerAddress property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5635"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the manufacturer address object (Read-only)
		/// </summary>
		public BOL.Address ManufacturerAddress
		{
			get
			{
				return this._manufacturerAddress;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5636"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>ManufacturerPhoneNumber property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5637"> 
		///		<ExpectedInput>Invalid string (15 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the Manufacturer Phone Number 
		/// </summary>
		public string ManufacturerPhoneNumber
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.ManufacturerPhoneNumber))
					return string.Empty;

				return this._drEquipmentRow[TABLE.Equipment.ManufacturerPhoneNumber].ToString();
			}
			set
			{
				if (value.Length > 14)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "14").ResString;
					throw new ArgumentException(errorMessage);
				}
				this._drEquipmentRow[TABLE.Equipment.ManufacturerPhoneNumber] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 14)));
			}
		}
	
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="89"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>DivisionCode property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="90"> 
		///		<ExpectedInput>Invalid string (6 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Division Code
		/// </summary>
		public string DivisionCode
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.DivisionCode))
					return Common.LogonUser.LogonUserDivisionCode;

				return this._drEquipmentRow[TABLE.Equipment.DivisionCode].ToString();
			}
			set
			{
				if (value == string.Empty)
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);
				if (value.Length > 5)
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);

				this._drEquipmentRow[TABLE.Equipment.DivisionCode] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5638"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>NextMaintenanceEvent property</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5639"> 
		///		<ExpectedInput>Invalid string (256 chars)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Next Maintenance Event
		/// </summary>
		public string NextMaintenanceEvent
		{
			get
			{
				if (this._drEquipmentRow.IsNull(TABLE.Equipment.NextMaintenanceText))
					return "";

				return this._drEquipmentRow[TABLE.Equipment.NextMaintenanceText].ToString();
			}
			set
			{
				if (value.Length > 255)
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);

				this._drEquipmentRow[TABLE.Equipment.NextMaintenanceText] = value;
				this.IsDirty = true;
				//
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length==0)||(value.Length > 255)));
			}
		}	


		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5714"> 
		///		<ExpectedInput>Valid serial number and manufacturer name string</ExpectedInput>
		///		<ExpectedOutput>Equipment guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5715"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty Guid</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Serach database for existing equipment matching serial number and manufacturer
		/// </summary>
		/// <param name="serialNumber">Serial Number</param>
		/// <param name="manufacturerName">Manufacturer Name</param>
		/// <returns></returns>
		public static Guid GetEquipmentBySerialNumberAndManufacturer(string serialNumber, string manufacturerName)
		{
			DataTable dt = DAL.Equipment.GetEquipmentBySerialNumberAndManufacturer(serialNumber, manufacturerName);
			//
			if (dt.Rows.Count>0)
			{
				return (Guid)dt.Rows[0][TABLE.Equipment.EquipmentGuid];
			}
			else
			{
				return Guid.Empty;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="91"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>Equipment Data Table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="92"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>Empty Data Table</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves equipment in inventory matching search parameters
		/// </summary>
		/// <returns>Equipment table</returns>
		public System.Data.DataTable GetEquipment()
		{
			return DAL.Equipment.GetEquipment(this.EquipmentType, this.ManufacturerName, this.SerialNumber);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="93"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Equipment ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="94"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Empty ArrayList</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve equipment in inventory with no search guidelines
		/// </summary>
		/// <returns>EquipmentGuid Arraylist</returns>
		public static System.Collections.ArrayList GetEquipmentList()
		{
			System.Collections.ArrayList equipmentList = new System.Collections.ArrayList();
			System.Data.DataTable dt = DAL.Equipment.GetEquipment();
			
			foreach(System.Data.DataRow dr in dt.Rows)
			{
				equipmentList.Add(new BOL.Equipment(dr));
				//equipmentList.Add((System.Guid)dr[TABLE.Equipment.EquipmentGuid]);
			}

			return(equipmentList);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5640"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Array of manufacturer names and equipment types</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5641"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get previously entered equipment types and manufacturers as an array of arrays
		/// </summary>
		/// <returns>Equipment type and Manufacturer name arrays</returns>
		public static System.Collections.ArrayList GetEquipmentDetails()
		{
			System.Collections.ArrayList equipTypeList = new System.Collections.ArrayList();
			System.Collections.ArrayList equipManufacturerList = new System.Collections.ArrayList();
			System.Collections.ArrayList equipData = new System.Collections.ArrayList();

			System.Data.DataSet ds = DAL.Equipment.GetEquipmentDetails();
			
			foreach(System.Data.DataRow dr in ds.Tables[0].Rows)
			{
				equipTypeList.Add(dr[TABLE.Equipment.EquipmentType].ToString());
			}

			foreach(System.Data.DataRow dr in ds.Tables[1].Rows)
			{
				equipManufacturerList.Add(dr[TABLE.Equipment.ManufacturerName].ToString());
			}

			equipData.Add(equipTypeList);
			equipData.Add(equipManufacturerList);

			return(equipData);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/14/2004</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads the local object from a data row parameter
		/// </summary>
		/// <param name="dataRow">Equipment data row</param>
		protected override void LoadFromDataRow(System.Data.DataRow dataRow)
		{
			this._drEquipmentRow = this._dtEquipment.NewRow();

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.Comments))
			{
				this._drEquipmentRow[TABLE.Equipment.Comments] = dataRow[TABLE.Equipment.Comments];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.DivisionCode))
			{
				this._drEquipmentRow[TABLE.Equipment.DivisionCode] = dataRow[TABLE.Equipment.DivisionCode];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.EquipmentGuid))
			{
				this._drEquipmentRow[TABLE.Equipment.EquipmentGuid] = dataRow[TABLE.Equipment.EquipmentGuid];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.EquipmentName))
			{
				this._drEquipmentRow[TABLE.Equipment.EquipmentName] = dataRow[TABLE.Equipment.EquipmentName];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.EquipmentType))
			{
				this._drEquipmentRow[TABLE.Equipment.EquipmentType] = dataRow[TABLE.Equipment.EquipmentType];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.InspectionIndicator))
			{
				this._drEquipmentRow[TABLE.Equipment.InspectionIndicator] = dataRow[TABLE.Equipment.InspectionIndicator];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.OutOfServiceIndicator))
			{
				this._drEquipmentRow[TABLE.Equipment.OutOfServiceIndicator] = dataRow[TABLE.Equipment.OutOfServiceIndicator];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.InvoiceNumber))
			{
				this._drEquipmentRow[TABLE.Equipment.InvoiceNumber] = dataRow[TABLE.Equipment.InvoiceNumber];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.LocalIdentifier))
			{
				this._drEquipmentRow[TABLE.Equipment.LocalIdentifier] = dataRow[TABLE.Equipment.LocalIdentifier];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.MaintenanceContractDate))
			{
				if (dataRow.IsNull(TABLE.Equipment.MaintenanceContractDate))
				{
					this._isMaintenanceContractDate = false;
					this._drEquipmentRow[TABLE.Equipment.MaintenanceContractDate] = DBNull.Value;
				}
				else
				{
					this._isMaintenanceContractDate = true;
					this._drEquipmentRow[TABLE.Equipment.MaintenanceContractDate] = dataRow[TABLE.Equipment.MaintenanceContractDate];
				}
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.MaintenanceContractNumber))
			{
				this._drEquipmentRow[TABLE.Equipment.MaintenanceContractNumber] = dataRow[TABLE.Equipment.MaintenanceContractNumber];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.ManufacturerName))
			{
				this._drEquipmentRow[TABLE.Equipment.ManufacturerName] = dataRow[TABLE.Equipment.ManufacturerName];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.NextMaintenanceText))
			{
				this._drEquipmentRow[TABLE.Equipment.NextMaintenanceText] = dataRow[TABLE.Equipment.NextMaintenanceText];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.ReceiptDate))
			{
				this._drEquipmentRow[TABLE.Equipment.ReceiptDate] = dataRow[TABLE.Equipment.ReceiptDate];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.ReceiptTechId))
			{
				this._drEquipmentRow[TABLE.Equipment.ReceiptTechId] = dataRow[TABLE.Equipment.ReceiptTechId];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.SerialNumber))
			{
				this._drEquipmentRow[TABLE.Equipment.SerialNumber] = dataRow[TABLE.Equipment.SerialNumber];
			}
			if (dataRow.Table.Columns.Contains(TABLE.Address.AddressLine1))
			{
				this.ManufacturerAddress.AddressLine1 = dataRow[TABLE.Address.AddressLine1].ToString();
			}

			if (dataRow.Table.Columns.Contains(TABLE.Address.AddressCity))
			{
				this.ManufacturerAddress.City = dataRow[TABLE.Address.AddressCity].ToString();
			}

			if (dataRow.Table.Columns.Contains(TABLE.Address.StateCode))
			{
				this.ManufacturerAddress.StateCode = dataRow[TABLE.Address.StateCode].ToString();
			}
			
			if (dataRow.Table.Columns.Contains(TABLE.Address.AddressZipCode))
			{
				this.ManufacturerAddress.ZipCode = dataRow[TABLE.Address.AddressZipCode].ToString();
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.ManufacturerPhoneNumber))
			{
				this._drEquipmentRow[TABLE.Equipment.ManufacturerPhoneNumber] = dataRow[TABLE.Equipment.ManufacturerPhoneNumber];
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.RowVersion))
			{
				if(!dataRow.IsNull(TABLE.Equipment.RowVersion))
				{
					this.RowVersion = (byte[])dataRow[TABLE.Equipment.RowVersion];
				}
			}

			this.IsDirty = true;

		}



		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="95"> 
		///		<ExpectedInput>Valid equipment data row</ExpectedInput>
		///		<ExpectedOutput>Equipment Data Row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="96"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads a datarow with the data from the object
		/// </summary>
		/// <param name="dataRow">Equipment data row schema</param>
		/// <returns>Populated equipment data row</returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dataRow)
		{
			if (dataRow.Table.Columns.Contains(TABLE.Equipment.Comments))
			{
				dataRow[TABLE.Equipment.Comments] = this.Comments;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.DivisionCode))
			{
				dataRow[TABLE.Equipment.DivisionCode] = this.DivisionCode;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.EquipmentGuid))
			{
				dataRow[TABLE.Equipment.EquipmentGuid] = this.EquipmentGuid;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.EquipmentName))
			{
				dataRow[TABLE.Equipment.EquipmentName] = this.EquipmentName;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.EquipmentType))
			{
				dataRow[TABLE.Equipment.EquipmentType] = this.EquipmentType;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.InspectionIndicator))
			{
				dataRow[TABLE.Equipment.InspectionIndicator] = this.InspectionIndicator;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.InvoiceNumber))
			{
				dataRow[TABLE.Equipment.InvoiceNumber] = this.InvoiceNumber;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.LocalIdentifier))
			{
				dataRow[TABLE.Equipment.LocalIdentifier] = this.LocalIdentifier;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.OutOfServiceIndicator))
			{
				dataRow[TABLE.Equipment.OutOfServiceIndicator] = this.OutOfServiceIndicator;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.MaintenanceContractDate))
			{
				if (this.IsMaintenanceContractDate)
				{
					dataRow[TABLE.Equipment.MaintenanceContractDate] = this.MaintenanceContractDate;
				}
				else
				{
					dataRow[TABLE.Equipment.MaintenanceContractDate] = DBNull.Value;
				}
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.MaintenanceContractNumber))
			{
				dataRow[TABLE.Equipment.MaintenanceContractNumber] = this.MaintenanceContractNumber;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.NextMaintenanceText))
			{
				dataRow[TABLE.Equipment.NextMaintenanceText] = this.NextMaintenanceEvent;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.ManufacturerName))
			{
				dataRow[TABLE.Equipment.ManufacturerName] = this.ManufacturerName;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.ReceiptDate))
			{
				dataRow[TABLE.Equipment.ReceiptDate] = this.ReceiptDate;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.ReceiptTechId))
			{
				dataRow[TABLE.Equipment.ReceiptTechId] = this.ReceiptTechId;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Equipment.SerialNumber))
			{
				dataRow[TABLE.Equipment.SerialNumber] = this.SerialNumber;
			}

			if (dataRow.Table.Columns.Contains(TABLE.Address.AddressGuid))
			{
				dataRow[TABLE.Address.AddressGuid] = this.ManufacturerAddress.AddressGuid;
			}

			if(dataRow.Table.Columns.Contains(TABLE.Equipment.ManufacturerPhoneNumber))
			{
				dataRow[TABLE.Equipment.ManufacturerPhoneNumber] = this.ManufacturerPhoneNumber;
			}

			if (!this.IsNew && (dataRow.Table.Columns.Contains(TABLE.Equipment.RowVersion)))
			{
				dataRow[TABLE.Equipment.RowVersion] = this.RowVersion;
			}

			return dataRow;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="97"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Empty Equipment DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="98"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a blank Equipment DataTable
		/// </summary>
		/// <returns>Eqwuipment data table</returns>
		public static System.Data.DataTable GetEmptyEquipmentTable()
		{
			return DAL.Equipment.GetEmptyEquipmentTable();
		}
	
		/// <summary>
		/// Set defaults in initial equipment table
		/// </summary>
		private void SetDefaultsInTable()
		{
			if(this._drEquipmentRow != null)
			{
				if(this._drEquipmentRow.Table.Columns.Contains(TABLE.Equipment.EquipmentName))
				{
					this._drEquipmentRow[TABLE.Equipment.EquipmentName] = string.Empty;
				}
				if(this._drEquipmentRow.Table.Columns.Contains(TABLE.Equipment.ReceiptDate))
				{
					this._drEquipmentRow[TABLE.Equipment.ReceiptDate] = System.DateTime.MinValue;
				}
				if(this._drEquipmentRow.Table.Columns.Contains(TABLE.Equipment.LocalIdentifier))
				{
					this._drEquipmentRow[TABLE.Equipment.LocalIdentifier] = string.Empty;
				}
				if(this._drEquipmentRow.Table.Columns.Contains(TABLE.Equipment.DivisionCode))
				{
					this._drEquipmentRow[TABLE.Equipment.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				}
				if(this._drEquipmentRow.Table.Columns.Contains(TABLE.Equipment.ReceiptTechId))
				{
					this._drEquipmentRow[TABLE.Equipment.ReceiptTechId] = Common.LogonUser.LogonUserName;
				}
				if(this._drEquipmentRow.Table.Columns.Contains(TABLE.Equipment.InspectionIndicator))
				{
					this._drEquipmentRow[TABLE.Equipment.InspectionIndicator] = true;
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5642"> 
		///		<ExpectedInput>Valid equipment object</ExpectedInput>
		///		<ExpectedOutput>success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5643"> 
		///		<ExpectedInput>Invalid equipment object with required fields set to null</ExpectedInput>
		///		<ExpectedOutput>Data Access Layer Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Insert equipment objects into the database
		/// </summary>
		/// <param name="equipments">Array list of equipment objects</param>
		/// <param name="lastUpdateFunctionId">Last update function</param>
		/// <returns>Success indicator</returns>
		public static bool InsertEquipment(ArrayList equipments, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable dtEquipment = BOL.Equipment.GetEmptyEquipmentTable();
			DataTable dtAddress = BOL.Address.GetEmptyAddressTableSchema();
			DataTable dtWorkloadEvents = null;
			
			//
			int nCtr = 0;
			System.Guid [] equipGuids = new Guid[equipments.Count];
			foreach(BOL.Equipment equipment in equipments)
			{
				dtEquipment.Rows.Add(equipment.LoadDataRowFromThis(dtEquipment.NewRow()).ItemArray);
				dtAddress.Rows.Add(equipment.ManufacturerAddress.LoadDataRowFromThis(dtAddress.NewRow()).ItemArray);
				
				equipGuids[nCtr] = (System.Guid) dtEquipment.Rows[nCtr][Common.VbecsTables.Equipment.EquipmentGuid];			
				
				nCtr = nCtr + 1;
			}
			dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.LoginEquipment, equipGuids, Common.WorkloadTransactionType.Miscellanious, false);
			return DAL.Equipment.Insert(dtEquipment,dtAddress, dtWorkloadEvents, lastUpdateFunctionId);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5644"> 
		///		<ExpectedInput>Valid equipment object</ExpectedInput>
		///		<ExpectedOutput>success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5645"> 
		///		<ExpectedInput>Invalid equipment object with row version set to null</ExpectedInput>
		///		<ExpectedOutput>Data Access Layer Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Update equipment object and, optionally, associated equipment maintenance
		/// </summary>
		/// <param name="equipmentMaintenance">Data table of equipment maintenance events</param>
		/// <param name="lastUpdateFunctionId">Lats update function</param>
		/// <returns>Success indicator</returns>
		public bool Update(DataTable equipmentMaintenance, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable dtEquipment = BOL.Equipment.GetEmptyEquipmentTable();
			DataTable dtAddress = BOL.Address.GetEmptyAddressTableSchema();
			DataTable dtInsertMaintenance = equipmentMaintenance.GetChanges(DataRowState.Added);
			DataTable dtUpdateMaintenance = equipmentMaintenance.GetChanges(DataRowState.Modified);
			System.Guid [] equipGuids = null;
			if(dtUpdateMaintenance != null)
			{
				equipGuids = new Guid[dtUpdateMaintenance.Rows.Count];
				int nCtr = 0;
				foreach(System.Data.DataRow dtRow in dtUpdateMaintenance.Rows)
				{
					equipGuids[nCtr] = (System.Guid) dtRow[Common.VbecsTables.Equipment.EquipmentGuid];
					nCtr = nCtr + 1;
				}
			}
			else
			{
				equipGuids = new Guid[1];
				equipGuids[0] = this.EquipmentGuid;
			}


			//
			if (this.IsDirty)
			{
				dtEquipment.Rows.Add(this.LoadDataRowFromThis(dtEquipment.NewRow()).ItemArray);
			}
			//
			if (this.ManufacturerAddress.IsDirty)
			{
				dtAddress.Rows.Add(this.ManufacturerAddress.LoadDataRowFromThis(dtAddress.NewRow()).ItemArray);
			}
			System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.UpdateEquipmentRecord, equipGuids, Common.WorkloadTransactionType.Miscellanious, false);

			//
			bool success = DAL.Equipment.Update(dtEquipment,dtAddress,dtInsertMaintenance,dtUpdateMaintenance, dtWorkloadEvents, lastUpdateFunctionId);
			//
			if (success)
			{
				equipmentMaintenance.AcceptChanges();
			}
			return success;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5646"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Array of equipment types</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5647"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get equipment types entered previously
		/// </summary>
		/// <returns>Array list of equipment type strings</returns>
		public static ArrayList GetEquipmentTypes()
		{
			DataTable dt = DAL.Equipment.GetEquipmentTypes();
			//
			ArrayList al = new ArrayList(dt.Rows.Count);
			//
			foreach(DataRow dr in dt.Rows)
			{
				al.Add(dr[TABLE.Equipment.EquipmentType].ToString());
			}
			//
			return al;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5648"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Array of manufacturer names</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5649"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get manufacturer names entered previously
		/// </summary>
		/// <returns>Array list of manufacturer name strings</returns>
		public static ArrayList GetEquipmentManufacturerNames()
		{
			DataTable dt = DAL.Equipment.GetEquipmentManufacturerNames();
			//
			ArrayList al = new ArrayList(dt.Rows.Count);
			//
			foreach(DataRow dr in dt.Rows)
			{
				al.Add(dr[TABLE.Equipment.ManufacturerName].ToString());
			}
			//
			return al;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5650"> 
		///		<ExpectedInput>Valid manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Data table of manufacturer address and phone numbers</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5651"> 
		///		<ExpectedInput>Unknown manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get addresses and phone numbers entered previously for a manufacturer
		/// </summary>
		/// <param name="manufacturerName">Manufacturer name</param>
		/// <returns>Data table of address and phone number info</returns>
		public static DataTable GetManufacturerAddresses(string manufacturerName)
		{
			return DAL.Equipment.GetManufacturerAddresses(manufacturerName);

		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5652"> 
		///		<ExpectedInput>Valid equipment guid</ExpectedInput>
		///		<ExpectedOutput>Equipment report data set</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5653"> 
		///		<ExpectedInput>Invalid equipment guid</ExpectedInput>
		///		<ExpectedOutput>0 record count</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get equipment report data set and format for SSRS
		/// </summary>
		/// <param name="equipmentGuid">Equipment unique identifier</param>
		/// <returns>Equipment and equipment maintenance data set</returns>
		public static DataSet GetEquipmentReport(Guid equipmentGuid)
		{
			DataSet dsReport = DAL.Equipment.GetEquipmentReport(equipmentGuid);
			//
			DataTable dtEquipment = dsReport.Tables[0];
			dtEquipment.TableName = TABLE.Equipment.TableName;
			dtEquipment.Columns.Add(ARTIFICIAL.MaintenanceRecordCount,typeof(int));
			dtEquipment.Columns.Add(ARTIFICIAL.EquipmentMirrorRecordCount,typeof(int));
			dtEquipment.Columns.Add(ARTIFICIAL.AddressMirrorRecordCount,typeof(int));
			//
			DataTable dtMaintain = dsReport.Tables[1];
			dtMaintain.TableName = TABLE.EquipmentMaintenance.TableName;
			//
			DataTable dtEquipmentMirror = dsReport.Tables[2];
			dtEquipmentMirror.TableName = TABLE.Equipment.TableName+"Mirror";
			//
			DataTable dtAddressMirror = dsReport.Tables[3];
			dtAddressMirror.TableName = TABLE.Address.TableName+"Mirror";
			//
			foreach(DataRow dr in dtEquipment.Rows)
			{
				dr[ARTIFICIAL.MaintenanceRecordCount] = dtMaintain.Rows.Count;
				dr[ARTIFICIAL.EquipmentMirrorRecordCount] = dtEquipmentMirror.Rows.Count;
				dr[ARTIFICIAL.AddressMirrorRecordCount] = dtAddressMirror.Rows.Count;
			}
			//
			return dsReport;
		}

		#endregion
	}
}


